<?php
/**
 * VIP-based rebate processing functions for betting system
 * Only users with VIP Level 1 or higher get rebates
 * Rebate rates are based on VIP levels
 */

/**
 * Get user's VIP level
 * @param mysqli $conn Database connection
 * @param int $userId User ID
 * @return int VIP level (0 if not VIP)
 */
function getUserVipLevel($conn, $userId) {
    $vipQuery = "SELECT lvl FROM vip WHERE userid = '$userId'";
    $vipResult = $conn->query($vipQuery);
    
    if ($vipResult && $vipResult->num_rows > 0) {
        $vipRow = $vipResult->fetch_assoc();
        return (int)$vipRow['lvl'];
    }
    
    return 0; // No VIP level
}

/**
 * Get rebate rate based on VIP level
 * @param int $vipLevel VIP level
 * @return float Rebate rate
 */
function getVipRebateRate($vipLevel) {
    $vipRates = [
        0 => 0.0000,  // 0% - No VIP
        1 => 0.0005,  // 0.05% - VIP 1
        2 => 0.0008,  // 0.08% - VIP 2
        3 => 0.0012,  // 0.12% - VIP 3
        4 => 0.0015,  // 0.15% - VIP 4
        5 => 0.0018,  // 0.18% - VIP 5
        6 => 0.0020,  // 0.20% - VIP 6
        7 => 0.0022,  // 0.22% - VIP 7
        8 => 0.0025,  // 0.25% - VIP 8
        9 => 0.0028,  // 0.28% - VIP 9
        10 => 0.0030  // 0.30% - VIP 10
    ];
    
    return $vipRates[$vipLevel] ?? 0; // Return 0 if not VIP
}

/**
 * Process rebate for a specific bet (VIP-based)
 * @param mysqli $conn Database connection
 * @param int $userId User ID
 * @param int $betId Bet ID
 * @param string $gameTable Game table name
 * @param float $betAmount Bet amount
 * @param string $betTime Bet timestamp
 * @return bool Success status
 */
function processBetRebate($conn, $userId, $betId, $gameTable, $betAmount, $betTime) {
    // Get user's VIP level
    $vipLevel = getUserVipLevel($conn, $userId);
    
    // Only process rebate if user is VIP level 1 or higher
    if ($vipLevel < 1) {
        return true; // No rebate for non-VIP users
    }
    
    // Get rebate rate based on VIP level
    $rebateRate = getVipRebateRate($vipLevel);
    
    if ($rebateRate == 0) {
        return true; // No rebate for this VIP level
    }
    
    // Calculate rebate amount
    $rebateAmount = $betAmount * $rebateRate;
    
    // Check if rebate already exists for this bet
    $checkQuery = "SELECT id FROM bet_rebates WHERE user_id = '$userId' AND bet_id = '$betId' AND game_table = '$gameTable'";
    $checkResult = $conn->query($checkQuery);
    
    if ($checkResult->num_rows == 0) {
        // Insert new rebate record
        $insertQuery = "INSERT INTO bet_rebates (user_id, bet_id, game_table, bet_amount, rebate_rate, rebate_amount, vip_level, created_at) 
                       VALUES ('$userId', '$betId', '$gameTable', '$betAmount', '$rebateRate', '$rebateAmount', '$vipLevel', '$betTime')";
        return $conn->query($insertQuery);
    }
    
    return true; // Already processed
}

/**
 * Create bet_rebates table if it doesn't exist (updated for VIP system)
 * @param mysqli $conn Database connection
 * @return bool Success status
 */
function createBetRebatesTable($conn) {
    $createTableQuery = "CREATE TABLE IF NOT EXISTS bet_rebates (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        bet_id INT NOT NULL,
        game_table VARCHAR(50) NOT NULL,
        bet_amount DECIMAL(10,2) NOT NULL,
        rebate_rate DECIMAL(10,6) NOT NULL,
        rebate_amount DECIMAL(10,2) NOT NULL,
        vip_level INT NOT NULL DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_created_at (created_at),
        INDEX idx_vip_level (vip_level),
        UNIQUE KEY unique_bet (user_id, bet_id, game_table)
    )";
    
    // Try to add vip_level column if it doesn't exist (for existing installations)
    $conn->query("ALTER TABLE bet_rebates ADD COLUMN vip_level INT NOT NULL DEFAULT 0");
    
    return $conn->query($createTableQuery);
}

/**
 * Get user's total rebate information (VIP-based)
 * @param mysqli $conn Database connection
 * @param int $userId User ID
 * @return array Rebate information
 */
function getUserRebateInfo($conn, $userId) {
    $today = date('Y-m-d');
    
    // Get user's current VIP level
    $vipLevel = getUserVipLevel($conn, $userId);
    $currentRebateRate = getVipRebateRate($vipLevel);
    
    // Get rebate totals
    $rebateQuery = "SELECT 
                    SUM(rebate_amount) as total_rebate,
                    SUM(CASE WHEN DATE(created_at) = '$today' THEN rebate_amount ELSE 0 END) as day_rebate
                    FROM bet_rebates 
                    WHERE user_id = '$userId'";
    $rebateResult = $conn->query($rebateQuery);
    $rebateData = $rebateResult->fetch_assoc();
    
    $totalRebateAmount = $rebateData['total_rebate'] ?? 0;
    $dayRebateAmount = $rebateData['day_rebate'] ?? 0;
    
    // Get individual bet records for wash list
    $washList = [];
    $washListQuery = "SELECT bet_id, bet_amount, rebate_rate, rebate_amount, created_at, game_table, vip_level 
                      FROM bet_rebates 
                      WHERE user_id = '$userId' 
                      ORDER BY created_at DESC 
                      LIMIT 50";
    $washListResult = $conn->query($washListQuery);
    
    if ($washListResult && $washListResult->num_rows > 0) {
        while ($washRow = $washListResult->fetch_assoc()) {
            $washList[] = [
                'washVolume' => (float)$washRow['bet_amount'],
                'washRate' => (float)$washRow['rebate_rate'],
                'rebateAmount' => (float)$washRow['rebate_amount'],
                'gameTable' => $washRow['game_table'],
                'betTime' => $washRow['created_at'],
                'betId' => $washRow['bet_id'],
                'vipLevel' => (int)$washRow['vip_level']
            ];
        }
    }
    
    return [
        'codeWashAmount' => 0,
        'dayRebate' => (float)$dayRebateAmount,
        'totalRebate' => (float)$totalRebateAmount,
        'washRate' => $currentRebateRate,
        'washList' => $washList,
        'vipLevel' => $vipLevel
    ];
}
?>
